#ifndef CYGONCE_DEVS_FLASH_CDRMONET_INL
#define CYGONCE_DEVS_FLASH_CDRMONET_INL

//==========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):    Bob Koninckx
// Contributors: Bob Koninckx
// Date:         2001-12-19
// Purpose:      
// Description:  Motorola CDR_MoneT flash driver
// Notes:
//
//####DESCRIPTIONEND####
//
//==========================================================================

#include <pkgconf/hal.h>
#include <pkgconf/devs_flash_motorola_cdrmonet.h>
#include <cyg/io/gmd.h>
#include CYGHWR_MEMORY_LAYOUT_H

#define _FLASH_PRIVATE_
#include <cyg/io/flash.h>

//----------------------------------------------------------------------------
// Now that device properties are defined, include magic for defining
// accessor type and constants.
#include <cyg/io/flash_dev.h>

//----------------------------------------------------------------------------
// Information about supported devices
typedef struct flash_dev_info {
  int        REVISION;
  cyg_uint32 PPCCMF300[66];
} flash_dev_info_t;

static const flash_dev_info_t * flash_dev_info = 0;
static tCMF_PART         * cmf_part            = 0; 
static tCMF_PROGRAM_DATA * cmf_program_data    = 0;
static tCMF_ERASE_DATA   * cmf_erase_data      = 0;

static const flash_dev_info_t supported_devices[] = {
#include <cyg/io/flash_cdrmonet_versions.inl>
};
#define NUM_DEVICES (sizeof(supported_devices)/sizeof(flash_dev_info_t))

// Maybe something extra is needed here ? The documentation is not really
// clear on its function
void
cmf_callback(void)
{
}

int 
flash_hwr_map_error(int e)
{
  return e;
}

//----------------------------------------------------------------------------
// Find out which core CMF revision we have
// The processor version tells us which flash 
// cmf core revision we have
int
cmf_query_revision(void)
{
#ifdef CYGPKG_HAL_POWERPC_MPC5xx
  cyg_uint32 immr;
  asm volatile ("mfimmr %0;": "=r" (immr));

  switch(immr & 0xffff0000)
  {
    case 0x30100000: // G    devices
    case 0x30200000: // K    devices
      return 5;
    case 0x30310000: // K2   devices
      return 6;
    case 0x30320000: // K3   devices
      return 7;
    case 0x30400000: // M    devices
      return 9;
    case 0x04000000: // A, C devices
    default:
      break;
  }
#endif
  // Unknown device
  return 0;
}

//----------------------------------------------------------------------------
// Map cmf errors to a package error
// 
int
cmf_map_initinfo(cyg_uint8 a_info, int a_cmf_core)
{
  if(a_cmf_core == 5)
  { // These may safely be ignored, CMF core 5 does not support censor
    // and the driver sometimes complains about that
    a_info &= ~((cyg_uint8)CMF_INFO_CENSOR_MODULE_A);
    a_info &= ~((cyg_uint8)CMF_INFO_CENSOR_MODULE_B);
  }

  if((a_info & CMF_INFO_CTRL_REGS_INIT_A) ||
     (a_info & CMF_INFO_CTRL_REGS_INIT_B))
    return FLASH_ERR_HWR;
  
  if((a_info & CMF_INFO_NO_VPP_A) ||
     (a_info & CMF_INFO_NO_VPP_B))
    return FLASH_ERR_PROGRAM_SUSPEND;
  
  if((a_info & CMF_INFO_CENSOR_MODULE_A) ||
     (a_info & CMF_INFO_CENSOR_MODULE_B))
    return FLASH_ERR_PROTOCOL;
  
  if(a_info & CMF_INFO_INVALID_ARRAY_BASE)
    return FLASH_ERR_INVALID;
  
  if(a_info & CMF_INFO_BACKUP_CLOCK)
    return FLASH_ERR_PROGRAM_SUSPEND;

  return FLASH_ERR_OK;
}

int
cmf_map_error(cyg_uint8 a_error)
{
  if(a_error == CMF_OK)
    return FLASH_ERR_OK;

  if(a_error & CMF_ERROR_PROTECTED_BLOCK)
    return FLASH_ERR_PROTECT;

  if(a_error & CMF_ERROR_SRC_DEST_VERIFY)
    return FLASH_ERR_DRV_VERIFY;

  if((a_error & CMF_ERROR_ALIGNMENT) ||
     (a_error & CMF_ERROR_ARRAY_RANGE))
    return FLASH_ERR_INVALID;

  if(a_error & CMF_ERROR_ARRAY_EMR_FAIL)
    return FLASH_ERR_ERASE;
  if(a_error & CMF_ERROR_SHADOW_EMR_FAIL)
    return FLASH_ERR_ERASE;

  if(a_error & CMF_ERROR_ARRAY_PMR_FAIL)
    return FLASH_ERR_PROGRAM;
  if(a_error & CMF_ERROR_SHADOW_PMR_FAIL)
    return FLASH_ERR_PROGRAM;

  if(a_error & CMF_ERROR_NOT_BLANK)
    return FLASH_ERR_PROTOCOL;
  if(a_error & CMF_ERROR_CENSOR_DEVICE_MODE)
    return FLASH_ERR_PROTOCOL;
  if(a_error & CMF_ERROR_CENSOR_MODULE)
    return FLASH_ERR_PROTOCOL;
  if(a_error & CMF_ERROR_CENSOR_INVALID_REQUEST)
    return FLASH_ERR_PROTOCOL;
  if(a_error & CMF_ERROR_CENSOR_CHANGE)
    return FLASH_ERR_PROTOCOL;
  if(a_error & CMF_ERROR_CENSOR_VALUE)
    return FLASH_ERR_PROTOCOL;
    
  if(a_error & CMF_ERROR_SHADOW_RANGE)
    return FLASH_ERR_INVALID;

  if(a_error & CMF_ERROR_SYSCLK_MISMATCH)
    return FLASH_ERR_HWR;

  // Keep the compiler happy and signal a generic error
  return FLASH_ERR_PROTOCOL;
}

// Translate an address to a block nr, page nr and page offset
bool
cmf_translate_address(void * a_address, tCMF_PART * a_part,
                      int * a_block, int * a_page, int * a_poff)
{
  cyg_uint32 offset_address;

  if(a_part->arrayBase > (cyg_uint32)a_address)
    return false;
  if((a_part->arrayBase + 0x6ffff) < (cyg_uint32)a_address)
    return false;

  offset_address = (cyg_uint32)a_address - a_part->arrayBase;

  switch(offset_address & 0xffff8000)
  {
    case 0x00000000:
      {
      *a_poff = (offset_address - 0x00000000)%64;
      *a_page = (offset_address - 0x00000000)/64;
      *a_block = 0;
      break;
      }
    case 0x00008000:
      {
      *a_poff = (offset_address - 0x00008000)%64;
      *a_page = (offset_address - 0x00008000)/64;
      *a_block = 1;
      break;
      }
    case 0x00010000:
      {
      *a_poff = (offset_address - 0x00010000)%64;
      *a_page = (offset_address - 0x00010000)/64;
      *a_block = 2;
      break;
      }
    case 0x00018000:
      {
      *a_poff = (offset_address - 0x00018000)%64;
      *a_page = (offset_address - 0x00018000)/64;
      *a_block = 3;
      break;
      }
    case 0x00020000:
      {
      *a_poff = (offset_address - 0x00020000)%64;
      *a_page = (offset_address - 0x00020000)/64;
      *a_block = 4;
      break;
      }
    case 0x00028000:
      {
      *a_poff = (offset_address - 0x00028000)%64;
      *a_page = (offset_address - 0x00028000)/64;
      *a_block = 5;
      break;
      }
    case 0x00030000:
      {
      *a_poff = (offset_address - 0x00030000)%64;
      *a_page = (offset_address - 0x00030000)/64;
      *a_block = 6;
      break;
      }
    case 0x00038000:
      {
      *a_poff = (offset_address - 0x00038000)%64;
      *a_page = (offset_address - 0x00038000)/64;
      *a_block = 7;
      break;
      }
    case 0x00040000:
      {
      *a_poff = (offset_address - 0x00040000)%64;
      *a_page = (offset_address - 0x00040000)/64;
      *a_block = 8;
      break;
      }
    case 0x00048000:
      {
      *a_poff = (offset_address - 0x00048000)%64;
      *a_page = (offset_address - 0x00048000)/64;
      *a_block = 9;
      break;
      }
    case 0x00050000:
      {
      *a_poff = (offset_address - 0x00050000)%64;
      *a_page = (offset_address - 0x00050000)/64;
      *a_block = 10;
      break;
      }
    case 0x00058000:
      {
      *a_poff = (offset_address - 0x00058000)%64;
      *a_page = (offset_address - 0x00058000)/64;
      *a_block = 11;
      break;
      }
    case 0x00060000:
      {
      *a_poff = (offset_address - 0x00060000)%64;
      *a_page = (offset_address - 0x00060000)/64;
      *a_block = 12;
      break;
      }
    case 0x00068000:
      {
      *a_poff = (offset_address - 0x00068000)%64;
      *a_page = (offset_address - 0x00068000)/64;
      *a_block = 13;
      break;
      }
    default:
      return false;
  }

  return true;
}

int
program_cmfcfig(cyg_uint32 a_cmfcfig)
{
  cyg_uint16 block_mask = 0x8000;
  cyg_uint32 data_buf[16];

  data_buf[0]  = a_cmfcfig;
  data_buf[1]  = 0xffffffff;
  data_buf[2]  = 0xffffffff;
  data_buf[3]  = 0xffffffff;
  data_buf[4]  = 0xffffffff;
  data_buf[5]  = 0xffffffff;
  data_buf[6]  = 0xffffffff;
  data_buf[7]  = 0xffffffff;
  data_buf[8]  = 0xffffffff;
  data_buf[9]  = 0xffffffff;
  data_buf[10] = 0xffffffff;
  data_buf[11] = 0xffffffff;
  data_buf[12] = 0xffffffff;
  data_buf[13] = 0xffffffff;
  data_buf[14] = 0xffffffff;
  data_buf[15] = 0xffffffff;
  
  if(!cmf_part)
    return FLASH_ERR_NOT_INIT;

  // Write the page buffer
  return cmf_map_error(ParallelProgram(cmf_part, cmf_program_data,
				       (cyg_uint8 *)&block_mask,
				       (cyg_uint32)data_buf,
				       0, // no offset
				       1, // One page at a time
				       1, // Shadow information
				       cmf_callback));
}

//----------------------------------------------------------------------------
// Initialize driver details
// 
int
flash_hwr_init(void)
{
  cyg_uint16 offset_cmf_part;
  cyg_uint16 offset_cmf_program_data;
  cyg_uint16 offset_cmf_erase_data;
  
  int i;
  int cmf_revision = cmf_query_revision();

  flash_dev_info = supported_devices;
  for(i = 0; i < NUM_DEVICES; i++)
  {
    if(flash_dev_info->REVISION == cmf_revision)
      break;
    flash_dev_info++;
  }

  // Did we find the device ? If not return error
  if(NUM_DEVICES == i)
    return FLASH_ERR_DRV_WRONG_PART;

  // All offsets are given in bytes
  offset_cmf_part         = ((cyg_uint16 *)(flash_dev_info->PPCCMF300))[GMDIO_OFFSET_PART_DESCRIPTION/2];
  offset_cmf_program_data = ((cyg_uint16 *)(flash_dev_info->PPCCMF300))[GMDIO_OFFSET_PROGRAM_DATA/2];
  offset_cmf_erase_data   = ((cyg_uint16 *)(flash_dev_info->PPCCMF300))[GMDIO_OFFSET_ERASE_DATA/2];
  
  if(offset_cmf_part != 0xffff)
    cmf_part = (tCMF_PART *)(&(flash_dev_info->PPCCMF300[offset_cmf_part/4]));
      
  if(offset_cmf_program_data != 0xffff)
    cmf_program_data = (tCMF_PROGRAM_DATA *)(&(flash_dev_info->PPCCMF300[offset_cmf_program_data/4]));
      
  if(offset_cmf_erase_data != 0xffff)
    cmf_erase_data = (tCMF_ERASE_DATA *)(&(flash_dev_info->PPCCMF300[offset_cmf_erase_data/4]));
      
  cmf_part->enabledBlocks[CMF_MODULE_A] = 0xFF; // enable all Blocks in module A
  cmf_part->enabledBlocks[CMF_MODULE_B] = 0xFC; // enable all Blocks in module B

  cmf_part->enableBDM = 0;                      // don't enter BDM after func exit

  // Hard wired. I don't like this
  flash_info.block_size = 32*1024;
  flash_info.blocks     = 14;
  flash_info.start = (void *)(cmf_part->arrayBase);
  flash_info.end = (void *)(cmf_part->arrayBase+32*1024*14);

  // call ParallelInit
  return cmf_map_initinfo(ParallelInit(cmf_part), cmf_revision);
}

//----------------------------------------------------------------------------
// See if a range of FLASH addresses overlaps currently running code
bool
flash_code_overlaps(void *start, void *end)
{
  extern unsigned char _stext[], _etext[];

  return ((((unsigned long)&_stext >= (unsigned long)start) &&
           ((unsigned long)&_stext < (unsigned long)end)) ||
          (((unsigned long)&_etext >= (unsigned long)start) &&
           ((unsigned long)&_etext < (unsigned long)end)));
}

//----------------------------------------------------------------------------
// Erase Block
// We silently assume that the requested size will never be greater than the
// block size of the device
int
flash_erase_block(void* block, unsigned int size)
{
  int block_nr, page_nr, page_off;
  cyg_uint16 block_mask = 0x8000;
  tCMF_COMP_DATA cmf_comp_data;

  // Four byte aligned
  cyg_uint32 tmp_address = ((cyg_uint32)block & 0xfffffff8);
  cyg_uint32 tmp_size    = size + (size % 4);
  
  if(!cmf_part)
    return FLASH_ERR_NOT_INIT;

  // If it is already blank, no need to erase is (and wearing the flash)
  if(BlankCheck(cmf_part, tmp_address, tmp_size, 
		0, &cmf_comp_data, cmf_callback) == CMF_OK)
    return FLASH_ERR_OK;
  
  // Not blank, erase it
  if(!cmf_translate_address(block, cmf_part, &block_nr, &page_nr, &page_off))
    return FLASH_ERR_INVALID;

  // Preserve the reset configuration word if the block nr-equals 0
  if(block_nr == 0)
  {
    cyg_uint32 cmfcfig;
    int err;

    // Enable shadow information
    *(cmf_part->cmf[0].cmfMCR) |= CMFMCR_SIE_MASK;
    cmfcfig = *((cyg_uint32 *)0x00000000);
    *(cmf_part->cmf[0].cmfMCR) &= ~(cyg_uint32)CMFMCR_SIE_MASK;

    err = cmf_map_error(ParallelErase(cmf_part, cmf_erase_data,
				      (cyg_uint8 *)&block_mask, cmf_callback));
    if(err == FLASH_ERR_OK)
      return program_cmfcfig(cmfcfig);
    else
      return err;      
  }
  
  // normal erase procedure
  block_mask = (block_mask >> block_nr);  
  return cmf_map_error(ParallelErase(cmf_part, cmf_erase_data, 
				     (cyg_uint8 *)&block_mask, cmf_callback));
}

//----------------------------------------------------------------------------
// Program Buffer
// We silently assume that the requested operation will never cross block
// boundaries 
int
flash_program_buf(void* addr, void* data, int len)
{
  int block_nr, page_nr, page_off;
  cyg_uint16 block_mask = 0x8000;
  int cnt;

  // We are going to write it page by page
  // this buffer guarantees 4 / 64 alignment
  cyg_uint32 data_buf[16];
  unsigned char * p = (unsigned char *)data_buf;
  unsigned char * q = (unsigned char *)data;

  const unsigned char * start = (unsigned char *)data;
  const unsigned char * stop  = (unsigned char *)data + len;

  int err = FLASH_ERR_OK;

  if(!cmf_part)
    return FLASH_ERR_NOT_INIT;

  if(!cmf_translate_address(addr, cmf_part, &block_nr, &page_nr, &page_off))
    return FLASH_ERR_INVALID;
  
  block_mask = (block_mask >> block_nr);
  q -= page_off;
  
  while(len && (err == FLASH_ERR_OK))
  {
    // Fill the page buffer
    for(cnt = 0; cnt < 64; cnt++)
    {
      if((q >= start) && (q < stop))
      {
        *p++ = *q++;
	len--;
      }
      else
      { // write a blank
	*p++ = 0xff;
	q++;
      }      
    }
    
    // Write the page buffer
    err = cmf_map_error(ParallelProgram(cmf_part, cmf_program_data,
					(cyg_uint8 *)&block_mask,
					(cyg_uint32)data_buf,
					page_nr * 64,
					1, // One page at a time
					0, // the main array,
					cmf_callback));

    // And start a new page
    p = (unsigned char *)data_buf;
    page_nr++;
  }  
  
  return err;
}

#endif // CYGONCE_DEVS_FLASH_CDRONET_INL
