//========================================================================
//
//      ppc_stub.c
//
//      Helper functions for stub, generic to all PowerPC processors
//
//========================================================================
//####COPYRIGHTBEGIN####
//                                                                          
// -------------------------------------------                              
// The contents of this file are subject to the Red Hat eCos Public License 
// Version 1.1 (the "License"); you may not use this file except in         
// compliance with the License.  You may obtain a copy of the License at    
// http://www.redhat.com/                                                   
//                                                                          
// Software distributed under the License is distributed on an "AS IS"      
// basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See the 
// License for the specific language governing rights and limitations under 
// the License.                                                             
//                                                                          
// The Original Code is eCos - Embedded Configurable Operating System,      
// released September 30, 1998.                                             
//                                                                          
// The Initial Developer of the Original Code is Red Hat.                   
// Portions created by Red Hat are                                          
// Copyright (C) 1998, 1999, 2000, 2001 Red Hat, Inc.                             
// All Rights Reserved.                                                     
// -------------------------------------------                              
//                                                                          
//####COPYRIGHTEND####
//========================================================================
//#####DESCRIPTIONBEGIN####
//
// Author(s):     Red Hat, jskov
// Contributors:  Red Hat, jskov, gthomas
// Date:          1998-08-20
// Purpose:       
// Description:   Helper functions for stub, generic to all PowerPC processors
// Usage:         
//
//####DESCRIPTIONEND####
//
//========================================================================

#include <stddef.h>

#include <pkgconf/hal.h>

#ifdef CYGDBG_HAL_DEBUG_GDB_INCLUDE_STUBS

#define CYGARC_HAL_COMMON_EXPORT_CPU_MACROS
#include <cyg/hal/ppc_regs.h>

#include <cyg/hal/hal_stub.h>
#include <cyg/hal/hal_arch.h>
#include <cyg/hal/hal_intr.h>

#ifdef CYGNUM_HAL_NO_VECTOR_TRACE
#define USE_BREAKPOINTS_FOR_SINGLE_STEP
#endif

#ifdef CYGDBG_HAL_DEBUG_GDB_THREAD_SUPPORT
#include <cyg/hal/dbg-threads-api.h>    // dbg_currthread_id
#endif

/* Given a trap value TRAP, return the corresponding signal. */

int __computeSignal (unsigned int trap_number)
{
    switch (trap_number)
    {
    case CYGNUM_HAL_VECTOR_MACHINE_CHECK:
        /* Machine check */
    case CYGNUM_HAL_VECTOR_DSI:
        /* Data access */
        return SIGSEGV;
      
    case CYGNUM_HAL_VECTOR_ISI:
        /* Instruction access (Ifetch) */
    case CYGNUM_HAL_VECTOR_ALIGNMENT:
        /* Data access */
        return SIGBUS;
                    
    case CYGNUM_HAL_VECTOR_INTERRUPT:
        /* External interrupt */
      return SIGINT;

    case CYGNUM_HAL_VECTOR_TRACE:
        /* Instruction trace */
        return SIGTRAP;
      
    case CYGNUM_HAL_VECTOR_PROGRAM:
#ifdef CYGPKG_HAL_POWERPC_PPC40x
        // The 40x is b0rken, returning 0 for these bits. Translate to
        // SIGTRAP to allow thread debugging.
        return SIGTRAP;
#else
        // The register PS contains the value of SRR1 at the time of
        // exception entry. Bits 11-15 contain information about the
        // cause of the exception. Bits 16-31 the PS (MSR) state.
#ifdef USE_BREAKPOINTS_FOR_SINGLE_STEP
        if (__is_single_step(get_register(PC))) {
            return SIGTRAP;
        }
#endif
        switch ((get_register (PS) >> 17) & 0xf){
        case 1:                         /* trap */
            return SIGTRAP;
        case 2:                         /* privileged instruction */
        case 4:                         /* illegal instruction */
            return SIGILL;
        case 8:                         /* floating point */
            return SIGFPE;
        default:                        /* should never happen! */
            return SIGILL;
        }            
#endif

    case CYGNUM_HAL_VECTOR_RESERVED_A:
    case CYGNUM_HAL_VECTOR_RESERVED_B:
        return SIGILL;

    case CYGNUM_HAL_VECTOR_FP_UNAVAILABLE:
        /* FPU disabled */
    case CYGNUM_HAL_VECTOR_FP_ASSIST:
        /* FPU assist */
        return SIGFPE;

    case CYGNUM_HAL_VECTOR_DECREMENTER:
        /* Decrementer alarm */
        return SIGALRM;

    case CYGNUM_HAL_VECTOR_SYSTEM_CALL:
        /* System call */
        return SIGSYS;

#if defined(CYGPKG_HAL_POWERPC_MPC8xx) || defined(CYGPKG_HAL_POWERPC_MPC5xx)
    case CYGNUM_HAL_VECTOR_SW_EMUL:
        /* A SW_EMUL is generated instead of PROGRAM for illegal
           instructions. */
        return SIGILL;

    case CYGNUM_HAL_VECTOR_DATA_BP:
    case CYGNUM_HAL_VECTOR_INSTRUCTION_BP:
    case CYGNUM_HAL_VECTOR_PERIPHERAL_BP:
    case CYGNUM_HAL_VECTOR_NMI:
        /* Developer port debugging exceptions. */
        return SIGTRAP;

#if defined(CYGPKG_HAL_POWERPC_MPC8xx)	
    case CYGNUM_HAL_VECTOR_ITLB_MISS:
    case CYGNUM_HAL_VECTOR_DTLB_MISS:
        /* Software reload of TLB required. */
        return SIGTRAP;
#endif
    case CYGNUM_HAL_VECTOR_ITLB_ERROR:
        /* Invalid instruction access. */
        return SIGBUS;

    case CYGNUM_HAL_VECTOR_DTLB_ERROR:
        /* Invalid data access. */
        return SIGSEGV;
#endif // defined(CYGPKG_HAL_POWERPC_MPC8xx)
        
    default:
        return SIGTERM;
    }
}


/* Return the trap number corresponding to the last-taken trap. */

int __get_trap_number (void)
{
    // The vector is not not part of the GDB register set so get it
    // directly from the save context.
    return _hal_registers->vector >> 8;
}

/* Set the currently-saved pc register value to PC. This also updates NPC
   as needed. */

void set_pc (target_register_t pc)
{
    put_register (PC, pc);
}


/*----------------------------------------------------------------------
 * Single-step support
 */

/* Set things up so that the next user resume will execute one instruction.
   This may be done by setting breakpoints or setting a single step flag
   in the saved user registers, for example. */

#ifdef USE_BREAKPOINTS_FOR_SINGLE_STEP

#if (HAL_BREAKINST_SIZE == 1)
typedef cyg_uint8 t_inst;
#elif (HAL_BREAKINST_SIZE == 2)
typedef cyg_uint16 t_inst;
#elif (HAL_BREAKINST_SIZE == 4)
typedef cyg_uint32 t_inst;
#else
#error "Don't know how to handle that size"
#endif

typedef struct
{
  t_inst *targetAddr;
  t_inst savedInstr;
} instrBuffer;

static instrBuffer sstep_instr[2];
static target_register_t irq_state = 0;

static void 
__insert_break(int indx, target_register_t pc)
{
    sstep_instr[indx].targetAddr = (t_inst *)pc;
    sstep_instr[indx].savedInstr = *(t_inst *)pc;
    *(t_inst*)pc = (t_inst)HAL_BREAKINST;
    __data_cache(CACHE_FLUSH);
    __instruction_cache(CACHE_FLUSH);
}

static void 
__remove_break(int indx)
{
    if (sstep_instr[indx].targetAddr != 0) {
        *(sstep_instr[indx].targetAddr) = sstep_instr[indx].savedInstr;
        sstep_instr[indx].targetAddr = 0;
        __data_cache(CACHE_FLUSH);
        __instruction_cache(CACHE_FLUSH);
    }
}

int
__is_single_step(target_register_t pc)
{
    return (sstep_instr[0].targetAddr == pc) ||
        (sstep_instr[1].targetAddr == pc);
}


// Compute the target address for this instruction, if the instruction
// is some sort of branch/flow change.

struct xl_form {
    unsigned int op : 6;
    unsigned int bo : 5;
    unsigned int bi : 5;
    unsigned int reserved : 5;
    unsigned int xo : 10;
    unsigned int lk : 1;
};

struct i_form {
    unsigned int op : 6;
    signed   int li : 24;
    unsigned int aa : 1;
    unsigned int lk : 1;
};

struct b_form {
    unsigned int op : 6;
    unsigned int bo : 5;
    unsigned int bi : 5;
    signed   int bd : 14;
    unsigned int aa : 1;
    unsigned int lk : 1;
};

union ppc_insn {
    unsigned int   word;
    struct i_form  i;
    struct b_form  b;
    struct xl_form xl;
};

static target_register_t
__branch_pc(target_register_t pc)
{
    union ppc_insn insn;

    insn.word = *(t_inst *)pc;

    // Decode the instruction to determine the instruction which will follow
    // Note: there are holes in this process, but the important ones work
    switch (insn.i.op) {
    case 16:
	/* bcx */
	if (insn.b.aa) {
	    return (target_register_t)(insn.b.bd << 2);
        } else {
	    return (target_register_t)((insn.b.bd << 2) + (long)pc);
        }
    case 18:
	/* bx */
	if (insn.i.aa) {
	    return (target_register_t)(insn.i.li << 2);
        } else {
	    return (target_register_t)((insn.i.li << 2) + (long)pc);
        }
    case 19:
	if (insn.xl.reserved == 0) {
	    if (insn.xl.xo == 528) {
		/* bcctrx */
                return (target_register_t)(get_register(CNT) & ~3);
	    } else if (insn.xl.xo == 16) {
		/* bclrx */
                return (target_register_t)(get_register(LR) & ~3);
	    }
	}
	break;
    default:
	break;
    }
    return (pc+4);
}

void __single_step(void)
{
    target_register_t msr = get_register(PS);
    target_register_t pc = get_register(PC);
    target_register_t next_pc = __branch_pc(pc);

    // Disable interrupts.
    irq_state = msr & MSR_EE;
    msr &= ~MSR_EE;
    put_register (PS, msr);

    // Set a breakpoint at the next instruction
    __insert_break(0, pc+4);
    if (next_pc != (pc+4)) {
        __insert_break(1, next_pc);
    }
}

/* Clear the single-step state. */

void __clear_single_step(void)
{
    target_register_t msr = get_register (PS);

    // Restore interrupt state.
    // FIXME: Should check whether the executed instruction changed the
    // interrupt state - or single-stepping a MSR changing instruction
    // may result in a wrong EE. Not a very likely scenario though.
    msr |= irq_state;

    // This function is called much more than its counterpart
    // __single_step.  Only re-enable interrupts if they where
    // disabled during the previous cal to __single_step. Otherwise,
    // this function only makes "extra sure" that no trace or branch
    // exception will happen.
    irq_state = 0;

    put_register (PS, msr);

    // Remove breakpoints
    __remove_break(0);
    __remove_break(1);
}

#else

static target_register_t irq_state = 0;

void __single_step (void)
{
    target_register_t msr = get_register (PS);

    // Set single-step flag in the exception context.
    msr |= (MSR_SE | MSR_BE);
    // Disable interrupts.
    irq_state = msr & MSR_EE;
    msr &= ~MSR_EE;

    put_register (PS, msr);
}

/* Clear the single-step state. */

void __clear_single_step (void)
{
    target_register_t msr = get_register (PS);

    // Clear single-step flag in the exception context.
    msr &= ~(MSR_SE | MSR_BE);
    // Restore interrupt state.
    // FIXME: Should check whether the executed instruction changed the
    // interrupt state - or single-stepping a MSR changing instruction
    // may result in a wrong EE. Not a very likely scenario though.
    msr |= irq_state;

    // This function is called much more than its counterpart
    // __single_step.  Only re-enable interrupts if they where
    // disabled during the previous cal to __single_step. Otherwise,
    // this function only makes "extra sure" that no trace or branch
    // exception will happen.
    irq_state = 0;

    put_register (PS, msr);
}
#endif

void __install_breakpoints (void)
{
    /* NOP since single-step HW exceptions are used instead of
       breakpoints. */
}

void __clear_breakpoints (void)
{
}


/* If the breakpoint we hit is in the breakpoint() instruction, return a
   non-zero value. */

int
__is_breakpoint_function ()
{
    return get_register (PC) == (target_register_t)&_breakinst;
}


/* Skip the current instruction.  Since this is only called by the
   stub when the PC points to a breakpoint or trap instruction,
   we can safely just skip 4. */

void __skipinst (void)
{
    put_register (PC, get_register (PC) + 4);
}

#endif // CYGDBG_HAL_DEBUG_GDB_INCLUDE_STUBS
